<?php
namespace chmContacts;

class Storage {
	private static $initialized = false;

	private static function init() {
		if (!self::$initialized) {
			self::$initialized = true;
		}
	}

	private static function convert($value, $format = 'string') {
		if ($format == 'string') {
			if (empty($value)) {
				return '';
			} else {
				return strval($value);
			}
		} else if ($format == 'date_Y-m-d') {
			return trim($value);
		} else {
			return $value;
		}
	}

	public static function getDataDir() {
		self::init();

		return Config::get('root_dir').'/'.Helper::normalizeRelativePath(Config::get('data_dir'));
	}

	public static function create() {
		self::init();

		global $wpdb;

		require_once(ABSPATH.'wp-admin/includes/upgrade.php');

		$dbVersion_option = get_option('chmContacts_db_version');
		$dbVersion_config = md5(file_get_contents(Config::get('db_tables')));

		//check for differing database version
		if ($dbVersion_option != $dbVersion_config) {
			foreach (Config::load('tables') as $sql) {
				dbDelta($sql);
			}
			//set current database version
			update_option('chmContacts_db_version', $dbVersion_config, false);
		}

		//create folder for contact images (corresponding to plug-in mediamatic)
		$term = wp_create_term(Config::get('images')['term'], Config::get('images')['taxonomy']);
		if (!is_wp_error($term)) {
			add_term_meta($term['term_id'], 'folder_type', 'default');
			add_term_meta($term['term_id'], 'folder_position', 9999);
		}
	}

	public static function addItem($input) {
		self::init();
		$added = true;
		$error = '';
		$itemTable = $input['item'];
		$name = trim($input['name']);
		$table = Config::table($itemTable);

		$check = Database::get(
			$table,
			[
				'name' => $name
			]
		);
		if (empty($check)) {
			Database::insert(
				$table,
				[
					'id' => Helper::id(),
					'name' => $name
				]
			);
		} else {
			$added = false;
			$error = 'Ein Eintrag mit dieser Bezeichnung existiert bereits.';
		}
		return [
			'added' => $added,
			'error' => $error
		];
	}

	//get list of items for given itemTable
	public static function getList($itemTable, $hideEmpty = false, $offset = 0, $limit = PHP_INT_MAX) {
		self::init();
		return Database::getRaw('
			SELECT *
			FROM
				`'.Config::table($itemTable).'`
			'.($hideEmpty ? '
				WHERE `id` IN (
					SELECT
						DISTINCT `item_id` FROM `'.Config::table('map_items').'`
					WHERE
						(
							`item_table` = "'.Database::escape($itemTable).'"
						) AND (
							`item_id` NOT IN (
								SELECT `id`
								FROM `'.Config::table('companies').'`
								WHERE `name` IN (
									"'.implode('","', Config::get('hiddenCompanies')).'"
								)
							)
						) AND (
							`item_id` NOT IN (
								SELECT `id`
								FROM `'.Config::table('units').'`
								WHERE `name` IN (
									"'.implode('","', Config::get('hiddenUnits')['global']).'"
								)
							)
						)
				)
			' : '').'
			ORDER BY
				`name` ASC
			LIMIT '.$offset.', '.$limit.'
			;
		');
	}

	//get list of new contacts
	public static function getNewContacts($limit = 9999999) {
		self::init();
		$template = @file_get_contents(Config::get('template_dir').'contactWelcomeItem.html');

		$query = '
			SELECT *
			FROM `'.Config::table('contacts').'`
			WHERE (
				(`show_image` = 1)
				AND
				(`image_post_id` > 0)
				AND
				(`start_date` <= "'.date("Y-m-d").'")
				AND
				(`start_date` >= "'.date("Y-m-d", (time()-(intval(Config::get('contact_new_days'))*24*60*60))).'")
			) AND (
				`id` IN (
					SELECT `contact_id`
					FROM `'.Config::table('map_items').'`
					WHERE (
						`item_table` = "units"
					) AND (
						`item_id` NOT IN (
							SELECT `id`
							FROM `'.Config::table('units').'`
							WHERE `name` IN (
								"'.implode('","', Config::get('hiddenUnits')['welcome']).'"
							)
						)
					)
				)
			) AND (
				`id` IN (
					SELECT `contact_id`
					FROM `'.Config::table('map_items').'`
					WHERE (
						`item_table` = "jobs"
					) AND (
						`item_id` NOT IN (
							SELECT `id`
							FROM `'.Config::table('jobs').'`
							WHERE `name` IN (
								"'.implode('","', Config::get('hiddenJobs')['welcome']).'"
							)
						)
					)
				)
			) AND (
				`id` IN (
					SELECT `contact_id`
					FROM `'.Config::table('map_items').'`
					WHERE (
						`item_table` = "companies"
					) AND (
						`item_id` IN (
							SELECT `id`
							FROM `'.Config::table('companies').'`
							WHERE `name` NOT IN (
								"'.implode('","', Config::get('hiddenCompanies')).'"
							)
						)
					)
				)
			) AND (
				`id` IN (
					SELECT `contact_id`
					FROM `'.Config::table('map_items').'`
					WHERE (
						`item_table` = "units"
					) AND (
						`item_id` IN (
							SELECT `id`
							FROM `'.Config::table('units').'`
							WHERE `name` NOT IN (
								"'.implode('","', Config::get('hiddenUnits')['global']).'"
							)
						)
					)
				)
			)
			ORDER BY '.(($limit < 3) ? 'RAND()' : '`start_date` DESC').'
			LIMIT '.$limit.';
		';


		$contacts = Database::getRaw($query);
		$list = [];
		$gender = 'none';

		if (count($contacts) > 0) {
			for($i = 0; $i < count($contacts); $i++) {
				$contact = self::autocompleteContact($contacts[$i], false);
				if ($i == 0) {
					$salutation = trim(strtolower($contact['salutation']));
					$gender = (($salutation == 'herr') ? 'male' : (($salutation == 'frau') ? 'female' : 'none'));
				}
				array_push(
					$list,
					Helper::replaceMarkers(
						$template,
						$contact
					)
				);
			}
		}

		return [
			'gender' => $gender,
			'list' => $list
		];

	}

	//update unit details
	public static function updateUnits($units) {
		self::init();
		foreach($units as $id => $fields) {
			Database::update(
				'units',
				[
					'phone_internal' => $fields['phone_internal']
				],
				[
					'id' => $id
				]
			);
		}
		return true;
	}


	//get list of all contacts
	public static function listContacts($offset = 0, $limit = PHP_INT_MAX, $withImages = true) {
		self::init();
		$offset = max(0, intval($offset));
		$limit = max(0, intval($limit));
		if ($limit < PHP_INT_MAX) {
			$newOffset = $offset + $limit;
			$limit++;
		}

		$contacts = Database::getRaw('SELECT * FROM `'.Config::table('contacts').'` ORDER BY `last_name` ASC, `first_name` ASC LIMIT '.$offset.','.$limit.';');

		$more = (($limit < PHP_INT_MAX) && (count($contacts) == $limit));
		if ($more) {
			array_pop($contacts);
		}

		for($i = 0; $i < count($contacts); $i++) {
			$contacts[$i] = self::autocompleteContact($contacts[$i], false, $withImages);
			$contacts[$i]['editable'] = ($contacts[$i]['editable'] == 1);
			$contacts[$i]['sync_exclude'] = ($contacts[$i]['sync_exclude'] == 1);
		}

		return [
			'contacts' => $contacts,
			'more' => $more,
			'offset' => $more ? $newOffset : 0
		];
	}

	//save modified contact data
	public static function saveContact($input) {
		self::init();

		$attribute = array_key_exists('attribute', $input) ? $input['attribute'] : '';
		$contact = $input['contact'];

		if ($attribute == 'sync_exclude') {
		
			$saved = true;
			$error = '';
			$contactID = $contact;
			$sync_exclude = !!$input['sync_exclude'];

			$checkContact = self::getContact($contactID, 'id', false);


			if (!empty($checkContact) && ($checkContact['id'] == $contactID) && !$checkContact['editable']) {


				Database::update(
					Config::table('contacts'),
					[
						'sync_exclude' => $sync_exclude
					],
					[
						'id' => $contactID
					]
				);


				if (!$sync_exclude) {
					$checkFiles = [];
					$dataID = '';

					//find new/imported/excluded data files
					foreach([
						'imported' => '.json.imported',
						'excluded' => '.json.excluded',
						'new' => '.json',
					] as $mode => $ext) {
						$checkFiles[$mode] = '';
						foreach(glob(self::getDataDir().'/persons/*'.$ext) as $dataFile) {
							$id = json_decode(@file_get_contents($dataFile), true)['id'];
							if (md5($id) == $contactID) {
								$dataID = $id;
								$checkFiles[$mode] = $dataFile;
								break;
							}
						}
					}

					//check for contact still available in sync list
					$idList = json_decode(@file_get_contents(self::getDataDir().'/list.json'), true);
					if (in_array($dataID, $idList)) {
						//re-activate most recent data file for next sync run
						if (empty($checkFiles['new'])) {
							$checkFiles['new'] = self::getDataDir().'/persons/'.$dataID.'.json';
							if (!empty($checkFiles['excluded'])) {
								rename($checkFiles['excluded'], $checkFiles['new']);
								if (!empty($checkFiles['imported'])) {
									@unlink($checkFiles['imported']);
								}
							} else if (!empty($checkFiles['imported'])) {
								rename($checkFiles['imported'], $checkFiles['new']);
							}
						}
					}
				}

	
			} else {
				$saved = false;
				$error = 'Das Attribut kann für diesen Kontakt nicht geändert werden.';
			}


		} else {

			$image = $input['image'];
			$items = $input['items'];
			$saved = true;
			$error = '';

			$checkUnique = self::getContact($contact['contact_number'], 'contact_number', false);

			if (empty($contact['id'])) {
				if ($checkUnique === false) {
					$newContact = [
						'id' => Helper::id(),
						'editable' => 1,
						'sync_exclude' => 0,
						'status' => 'active',
						'image_post_id' => 0,
						'show_image' => 1
					];
					foreach (Config::get('contact_save_fields') as $field) {
						$newContact[$field] = $contact[$field];
					}
					Database::insert(
						Config::table('contacts'),
						$newContact
					);
					$contact['id'] = $newContact['id'];
				} else {
					$error = 'Der Kontakt kann nicht angelegt werden, weil die eingebebene Personalnummer bereits von einem anderen Kontakt verwendet wird.';
					$saved = false;
				}
			}
			if ($saved) {
				$checkContact = self::getContact($contact['id'], 'id', false);
				if (($checkContact !== false) && ($checkContact['editable'] || $checkContact['sync_exclude'])) {
					if (($checkUnique === false) || ($checkUnique['id'] === $checkContact['id'])) {
						$update = [];
						foreach (Config::get('contact_save_fields') as $field) {
							$update[$field] = $contact[$field];
						}
						if ($image['changed'] && $checkContact['editable']) { //no image modifications for sync_exclude
							if (self::deleteContactImage($contact['id'], true)) {
								$update['image_post_id'] = empty($image['src']) ? 0 : self::storeContactImage($contact, $image['src']);
							}
						}
						if ($checkContact['sync_exclude']) {
							unset($update['contact_number']);
						}
						Database::update(
							Config::table('contacts'),
							$update,
							[
								'id' => $contact['id']
							]
						);


						//delete all existing item links
						Database::delete(
							'map_items',
							[
								'contact_id' => $contact['id']
							]
						);

						foreach($items as $table => $itemIDs) {
							foreach ($itemIDs as $itemID) {
								//link item to contact
								Database::insert(
									'map_items',
									[
										'id' => Helper::id(),
										'item_table' => $table,
										'item_id' => $itemID,
										'contact_id' => $contact['id']
									]
								);
							}
						}

					} else {
						$error = 'Die eingebebene Personalnummer wird bereits von einem anderen Kontakt verwendet.';
						$saved = false;
					}
				} else {
					$error = 'Der Kontakt existiert nicht oder ist nicht bearbeitbar.';
					$saved = false;
				}
			}

			$contactID = $contact['id'];
		}

		return [
			'saved' => $saved,
			'error' => $error,
			'contactID' => $contactID,
			'updated' => $update
		];
	}



	public static function searchContactsByItem($category, $id, $offset = 0, $limit = PHP_INT_MAX) {
		self::init();

		$offset = max(0, intval($offset));
		$limit = max(0, intval($limit));
		if ($limit < PHP_INT_MAX) {
			$newOffset = $offset + $limit;
			$limit++;
		}

		$result = [];

		if (array_key_exists($category, Config::get('mappings'))) {
			$contacts = Database::getRaw('
				SELECT *
				FROM `'.Config::table('contacts').'`
				WHERE (
					(`start_date` <= "'.date("Y-m-d").'")
					OR
					(`start_date` IS NULL)
				) AND (
					`id` IN (
						SELECT `contact_id`
						FROM `'.Config::table('map_items').'`
						WHERE (
							`item_table` = "'.Database::escape($category).'"
						) AND (
							`item_id` = "'.Database::escape($id).'"
						)
					)
				) AND (
					`id` NOT IN (
						SELECT `contact_id`
						FROM `'.Config::table('map_items').'`
						WHERE (
							`item_table` = "companies"
						) AND (
							`item_id` IN (
								SELECT `id`
								FROM `'.Config::table('companies').'`
								WHERE `name` IN (
									"'.implode('","', Config::get('hiddenCompanies')).'"
								)
							)
						)
					)
				) AND (
					`id` NOT IN (
						SELECT `contact_id`
						FROM `'.Config::table('map_items').'`
						WHERE (
							`item_table` = "units"
						) AND (
							`item_id` IN (
								SELECT `id`
								FROM `'.Config::table('units').'`
								WHERE `name` IN (
									"'.implode('","', Config::get('hiddenUnits')['global']).'"
								)
							)
						)
					)
				)
				ORDER BY `room` ASC, `last_name` ASC, `first_name` ASC
				LIMIT '.$offset.','.$limit.'
			;');

			$more = (($limit < PHP_INT_MAX) && (count($contacts) == $limit));
			if ($more) {
				array_pop($contacts);
			}

			$result = [];
			foreach ($contacts as $contact) {
				array_push($result, self::autocompleteContact($contact));
			}
		}


		return [
			'contacts' => $result,
			'more' => $more,
			'offset' => $more ? $newOffset : 0
		];
		return $result;
	}


	public static function searchContacts($keywords, $offset = 0, $limit = PHP_INT_MAX) {
		self::init();

		$offset = max(0, intval($offset));
		$limit = max(0, intval($limit));
		if ($limit < PHP_INT_MAX) {
			$newOffset = $offset + $limit;
			$limit++;
		}

		if (!is_array($keywords)) {
			$keywords = explode(' ', $keywords);
		}

		$searchFields = Config::get('contact_search_fields');
		$searchItems = Config::get('contact_search_items');

		// $whereSet = [];

		$whereSet = [];

		foreach ($keywords as $keyword) {
			if (strlen($keyword) >= 3) {


				//check fields in contacts table for keyword match
				$where = [];
				foreach ($searchFields as $field) {
					array_push($where, '(`'.Database::escape($field).'` LIKE "%'.Database::escape($keyword).'%")');
				}

				//get contact ids from matched tables if name of match is mathing the keyword
				$subWhere = [];
				foreach ($searchItems as $itemTable) {
					array_push($subWhere, '
						(
							`id` IN (
								SELECT `contact_id` AS id
								FROM `'.Config::table('map_items').'`
								WHERE (
									`item_table` = "'.Database::escape($itemTable).'"
								) AND (
									`item_id` IN (
										SELECT `id`
										FROM `'.Config::table($itemTable).'`
										WHERE (
											`name` LIKE "%'.Database::escape($keyword).'%"
										)
									)
								)
							)
						)
					');
				}

				array_push($whereSet, '
					`id` IN (
						SELECT `id`
						FROM `'.Config::table('contacts').'`
						WHERE (
							('.implode(' OR ', $where).')
							OR '.implode(' OR ', $subWhere).'
						)
					)
				');


				// array_push($selects, '(`id` IN ('.$query.'))');

				// echo $query.PHP_EOL.PHP_EOL.'------------------------------------------------------'.PHP_EOL.PHP_EOL.PHP_EOL;

			}

		}

		// $query = implode(' UNION ', $selects);

		if (count($whereSet) > 0) {
			$query = '
				SELECT *
				FROM `'.Config::table('contacts').'`
				WHERE (
					(`start_date` <= "'.date("Y-m-d").'")
					OR
					(`start_date` IS NULL)
				) AND (
					'.implode(' AND ', $whereSet).'
				) AND (
					`id` IN (
						SELECT `contact_id`
						FROM `'.Config::table('map_items').'`
						WHERE (
							`item_table` = "companies"
						) AND (
							`item_id` IN (
								SELECT `id`
								FROM `'.Config::table('companies').'`
								WHERE `name` NOT IN (
									"'.implode('","', Config::get('hiddenCompanies')).'"
								)
							)
						)
					)
				) AND (
					`id` IN (
						SELECT `contact_id`
						FROM `'.Config::table('map_items').'`
						WHERE (
							`item_table` = "units"
						) AND (
							`item_id` IN (
								SELECT `id`
								FROM `'.Config::table('units').'`
								WHERE `name` NOT IN (
									"'.implode('","', Config::get('hiddenUnits')['global']).'"
								)
							)
						)
					)
				)
				ORDER BY `room` ASC, `last_name` ASC, `first_name` ASC
				LIMIT '.$offset.','.$limit.'
			';

			// echo $query;

			$contacts = Database::getRaw($query);
		} else {
			$contacts = [];
		}

		$more = (($limit < PHP_INT_MAX) && (count($contacts) == $limit));
		if ($more) {
			array_pop($contacts);
		}

		$result = [];
		foreach ($contacts as $contact) {
			array_push($result, self::autocompleteContact($contact));
		}

		return [
			'contacts' => $result,
			'more' => $more,
			'offset' => $more ? $newOffset : 0
		];
	}

	public static function getContact($value, $field = 'id', $publicFieldsOnly = true, $withImages = true) {
		self::init();
		if (!in_array($field, ['id', 'contact_number'])) {
			$field = 'id';
		}
		// $where = ($field == 'id')
		// 	? [
		// 		$field => $value
		// 	]
		// 	: [
		// 		[$field, 'LIKE', '%'.$value.'%']
		// 	];

		$contact = Database::get(
			'contacts',
			[$field => $value]
			// $where
		);
		if (!empty($contact) && (($contact['start_date'] <= date("Y-m-d")) || (empty($contact['start_date'])))) {
			return self::autocompleteContact($contact, $publicFieldsOnly, $withImages);
		}
		return false;
	}

	/*** get image urls of contact ***/
	private static function loadImageUrls($image_post_id) {
		$imageUrls = [];

		//retrieve images
		foreach(Config::get('image_sizes') as $key => $imageSize) {
			if ($image_post_id > 0) {
				$imageUrl = wp_get_attachment_image_url($image_post_id, $imageSize['id']);
			}
			if (empty($imageUrl)) {
				$imageUrl = false;
			}
			$imageUrls[$key] = $imageUrl;
		}

		return $imageUrls;
	}

	/*** add all contact details ***/
	private static function autocompleteContact($contact, $publicFieldsOnly = true, $withImages = true) {
		//prepare mappings
		$links = [];
		foreach(Config::get('mappings') as $table => $mapping) {
			//map items in alphabetical order
			$contact[$table] = Database::getRaw(
				'SELECT * FROM `'.Config::table($table).'` WHERE `id` IN (SELECT `item_id` FROM `'.Config::table('map_items').'` WHERE `item_table` = "'.Database::escape($table).'" AND `contact_id` = "'.$contact['id'].'") ORDER BY `name` ASC;'
			);
			$itemList = [];
			foreach ($contact[$table] as $item) {
				array_push($itemList, $item['name']);
			}
			$contact[$table.'List'] = implode(', ', $itemList);
			$contact[$table.'Count'] = count($itemList);
		}

		if ($withImages) {
			foreach (self::loadImageUrls($contact['image_post_id']) as $key => $imageUrl) {
				$contact[$key] = $imageUrl;
			}
		} else {
			foreach(Config::get('image_sizes') as $key => $imageSize) {
				$contact[$key] = null;
			}
		}

		if ($publicFieldsOnly) {
			$resultContact = [];
			foreach(Config::get('contact_public_fields') as $field) {
				$resultContact[$field] = $contact[$field];
			}
			return $resultContact;
		} else {
			return $contact;
		}
	}

	public static function getImageUrls($contactID) {
		self::init();

		$contact = self::getContact($contactID, 'id', false, false);

		if (!empty($contact)) {
			return self::loadImageUrls($contact['image_post_id']);
		} else {
			return [];
		}
			
	}

	public static function deleteContact($contactID, $onlyEditable = false) {
		self::init();
		if ($onlyEditable) {
			$contact = self::getContact($contactID, 'id', false);
			if (($contact !== false) && (!$contact['editable'])) {
				return false;
			}
		}
		//delete image
		self::deleteContactImage($contactID);
		//delete contact entry
		Database::delete(
			'contacts',
			[
				'id' => $contactID
			]
		);
		//delete all existing item links
		Database::delete(
			'map_items',
			[
				'contact_id' => $contactID
			]
		);
		return true;
	}

	public static function deleteContactImage($contactID, $onlyEditable = false) {
		self::init();
		//find already existing contact
		$existingContact = Database::get(
			'contacts',
			[
				'id' => $contactID
			]
		);
		//delete image if contact already exists
		if (!empty($existingContact)) {
			if ($onlyEditable && (!$existingContact['editable'])) {
				return false;
			}
			$imageID = intval($existingContact['image_post_id']);
			if ($imageID > 0) {
				wp_delete_post($imageID, true);
			}
		}
		return true;
	}

	public static function import($contactRaw) {
		self::init();
		if (!is_array($contactRaw)) {
			return false;
		}

		//empty dependant fields if master field is not empty
		foreach(Config::get('emptyFields') as $fieldName => $fieldsToEmpty) {
			if (!empty(trim($contactRaw[$fieldName]))) {
				foreach ($fieldsToEmpty as $emptyField) {
					$contactRaw[$emptyField] = '';
				}
			}
		}

		$contactID = md5($contactRaw['id']);

		$checkContact = self::getContact($contactID, 'id', false);

		if (!empty($checkContact) && ($checkContact['sync_exclude'] == 1)) {
			return 'excluded';
		}

		//prepare data from raw data
		$contact = [
			'id' => $contactID,
			'editable' => (array_key_exists('editable', $contactRaw) ? intval($contactRaw['editable']) : 0),
			'sync_exclude' => 0,
			'contact_number' => self::convert($contactRaw['personalnummer']),
			'salutation' => self::convert($contactRaw['anrede']),
			'title' => self::convert($contactRaw['titel']),
			'name_prefix' => self::convert($contactRaw['namensvorsatz']),
			'first_name' => self::convert($contactRaw['vorname']),
			'last_name' => self::convert($contactRaw['name']),
			'room' => self::convert($contactRaw['room']),
			'status' => 'active',
			'email' => self::convert($contactRaw['email']),
			'image_post_id' => 0,
			'show_image' => (intval($contactRaw['keine_einwilligung_foto']) == 0) ? 1 : 0,
			'phone_internal' => self::convert($contactRaw['telefon']),
			'phone_dect' => self::convert($contactRaw['dect']),
			'fax' => self::convert($contactRaw['fax']),
			'start_date' => self::convert($contactRaw['eintritt'], 'date-Y-m-d')
		];

		//delete image
		self::deleteContactImage($contactID);

		//insert/update contact data
		Database::replace(
			'contacts',
			$contact
		);

		//prepare mappings
		$links = [];
		foreach(Config::get('mappings') as $table => $mapping) {
			$items = [];
			foreach($mapping['fields'] as $field) {
				$value = trim($contactRaw[$field]);
				if (!empty($value)) {
					if ($mapping['type'] == 'csv') {
						$values = explode(',', $value);
						foreach ($values as $item) {
							$item = trim($item);
							if (!empty($item)) {
								array_push($items, $item);
							}
						}
					} else {
						array_push($items, $value);
					}
				}
			}
			$links[$table] = $items;
		}

		//delete all existing item links
		Database::delete(
			'map_items',
			[
				'contact_id' => $contactID
			]
		);

		$contactMappings = [];

		foreach($links as $table => $items) {

			$contactMappings[$table] = [];

			foreach ($items as $name) {

				//change unit name before import
				if ($table == 'units') {
					if ($name == 'Verwaltungsleitung') {
						$name = 'Verwaltung';
					}
				}

				//get existing item
				$item = Database::get(
					$table,
					[
						'name' => $name
					]
				);
				//add item if it doesn't exist
				if (empty($item)) {
					$item = [
						'id' => md5(uniqid()),
						'name' => $name
					];
					Database::insert(
						$table,
						$item
					);
				}
				//keep mapping temporary
				array_push($contactMappings[$table], $item['id']);
				//link item to contact
				Database::insert(
					'map_items',
					[
						'id' => md5(uniqid()),
						'item_table' => $table,
						'item_id' => $item['id'],
						'contact_id' => $contactID
					]
				);
			}
		}

		//fill empty internal phone with phone of first mapped unit (Abteilungs-Telefonnummer)
		if (empty($contact['phone_internal'])) {
			$unit = Database::get(
				'units',
				[
					'id' => $contactMappings['units'][0]
				]
			);
			if (!empty($unit)) {
				Database::update(
					'contacts',
					[
						'phone_internal' => $unit['phone_internal']
					],
					[
						'id' => $contactID
					]
				);
			}
		}

		if ( ($contact['show_image'] == 1) && !empty($contactRaw['foto']) ) {
			self::storeContactImage($contact, $contactRaw['foto']);
		}

		return true;

	}

	private static function storeContactImage($contact, $url) {
		$imageName = [];
		foreach (['last_name', 'first_name'] as $part) {
			if (!empty($contact[$part])) {
				array_push($imageName, $contact[$part]);
			}
		}
		$imageID = self::downloadImage($url, 'chmContacts_'.$contact['id'].'.'.self::getExtensionFromDataUrl($url), implode(', ', $imageName));
		Database::update(
			'contacts',
			[
				'image_post_id' => $imageID
			],
			[
				'id' => $contact['id']
			]
		);
		return $imageID;
	}

	//download image from url and store with given filename and optional description
	public static function downloadImage($url, $filename, $description = null) {
		self::init();
		$file = [
			'name' => $filename,
			'tmp_name' => (substr($url, 0, 5) != 'data:') ? download_url($url) : self::storeDataUrl($url)
		];

		$imageID = media_handle_sideload($file, 0, $description);

		if (!is_wp_error($imageID)) {

			$term = get_term_by('name', Config::get('images')['term'], Config::get('images')['taxonomy']);

			wp_set_object_terms($imageID, intval($term->term_id), Config::get('images')['taxonomy'], false);

		} else {
			$imageID = 0;
		}

		@unlink($file['tmp_name']);

		return $imageID;
	}

	//store image in data url as temporary file
	public static function storeDataUrl($url) {
		self::init();
		$filename = get_temp_dir().'chmContacts_contact_image.tmp';
		$image = explode(',', $url, 2)[1];
		if (!empty($image)) {
			$file = fopen($filename, 'wb');
			fwrite($file, base64_decode($image));
			fclose($file);
			chmod($filename, 0777);
		}
		return $filename;
	}

	//extract mime type from data url and return matching file extension
	public static function getExtensionFromDataUrl($data) {
		self::init();
		$data = @explode(';', $data, 2)[0];
		$data = @explode(':', $data, 2)[1];
		$extension = Config::get('mime_types')[$data];
		if (empty($extension)) {
			$extension = 'image';
		}
		return $extension;
	}

}
